/************************************************************************
 * NAME:	dir
 *
 * DESCR:	Directory functions.
 *
 * NOTES:	
 ************************************************************************/
#include "hdos.h"
#include "../fs-utility.h"


/************************************************************************
 * NAME:	hdos_dir_entry_unpack() & _pack()
 *
 * DESCR:	Un/packs a directory entry in the given buffer, and fills
 *		in a dirent structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
hdos_dir_entry_unpack(struct hdos_dirent *entry, unsigned char *buffer)
{
    memcpy(entry->name,&buffer[0],8);
    memcpy(entry->ext,&buffer[8],3);
    entry->project = buffer[11];
    entry->version = buffer[12];
    entry->cluster = buffer[13];
    entry->flags = buffer[14];

    entry->is_system = (HDOS_FLAGS_SYSTEM(entry->flags))?TRUE:FALSE;
    entry->is_locked = (HDOS_FLAGS_LOCKED(entry->flags))?TRUE:FALSE;
    entry->is_wp =     (HDOS_FLAGS_WP(entry->flags))?TRUE:FALSE;
    entry->is_contig = (HDOS_FLAGS_CONTIG(entry->flags))?TRUE:FALSE;

    entry->reserved = buffer[15];
    entry->firstgroup = buffer[16];
    entry->lastgroup = buffer[17];
    entry->lastsecidx = buffer[18];

    hdos_date_unpack(hdos_int_unpack(&buffer[19]),&entry->create);
    hdos_date_unpack(hdos_int_unpack(&buffer[21]),&entry->mod);

    return(TRUE);
}
 
static int
hdos_dir_entry_pack(struct hdos_dirent *entry, unsigned char *buffer)
{
    memcpy(&buffer[0],entry->name,8);
    memcpy(&buffer[8],entry->ext,3);
    buffer[11] = entry->project & 0x00ff;
    buffer[12] = entry->version & 0x00ff;
    buffer[13] = entry->cluster & 0x00ff;

    entry->flags = (entry->is_system)?HDOS_FLAGS_SYSTEM_BITS:0;
    entry->flags |= (entry->is_locked)?HDOS_FLAGS_LOCKED_BITS:0;
    entry->flags |= (entry->is_wp)?HDOS_FLAGS_WP_BITS:0;
    entry->flags |= (entry->is_contig)?HDOS_FLAGS_CONTIG_BITS:0;

    buffer[14] = entry->flags & 0x00ff;

    buffer[15] = entry->reserved & 0x00ff;
    buffer[16] = entry->firstgroup & 0x00ff;
    buffer[17] = entry->lastgroup & 0x00ff;
    buffer[18] = entry->lastsecidx & 0x00ff;

    hdos_int_pack(hdos_date_pack(&entry->create),&buffer[19]);
    hdos_int_pack(hdos_date_pack(&entry->mod),&buffer[21]);

    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_dir_block_unpack ()
 *
 * DESCR:	Pack/unpack a directory block.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- buffer should point to a 512-byte block
 *		- unpacks all of the directory entries too.
 ************************************************************************/
static int
hdos_dir_block_unpack(struct hdos_dirblock *dirblock, unsigned char *buffer)
{
    int	i;

    dirblock->entrylen = buffer[507];			/* will be 23	*/
    dirblock->this = hdos_int_unpack(&buffer[508]);
    dirblock->next = hdos_int_unpack(&buffer[510]);

    dirblock->entries = 512 / dirblock->entrylen;	/* will be 22	*/

    for (i=0; i < dirblock->entries; i++) {
	if (!hdos_dir_entry_unpack(&dirblock->entry[i],buffer + (i*dirblock->entrylen))) {
	    return(FALSE);
	}
    }
    return(TRUE);
}

static int
hdos_dir_block_pack(struct hdos_dirblock *dirblock, unsigned char *buffer)
{
    int	i;

    buffer[507] = dirblock->entrylen & 0x00ff;
    hdos_int_pack(dirblock->this,&buffer[508]);
    hdos_int_pack(dirblock->next,&buffer[510]);

    for (i=0; i < dirblock->entries; i++) {
	if (!hdos_dir_entry_pack(&dirblock->entry[i],buffer + (i*dirblock->entrylen))) {
	    return(FALSE);
	}
    }
    return(TRUE);
}	

/************************************************************************
 * NAME:	hdos_dir_block_alloc()
 *
 * DESCR:	Allocates new block storage if necessary.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if OK, FALSE otherwise
 *
 * NOTES:	- staticly set to 10 blocks at a time
 ************************************************************************/
static int
hdos_dir_blocks_alloc(struct hdos_dir *hdir)
{
    int	needed = hdir->blocks;
    int	alloc = hdir->maxblocks;
    int ssize = sizeof(struct hdos_dirblock);

    if (hdir->blocks > hdir->maxblocks) {	/* need more space	*/
	hdir->maxblocks += 10;
	hdir->block = (struct hdos_dirblock *)
	    realloc(hdir->block,sizeof(struct hdos_dirblock) * hdir->maxblocks);
    }

    if (hdir->block == NULL) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_dir_read() & hdos_dir_write()
 *
 * DESCR:	Reads the directory on the current disk, and stores it in
 *		the directory buffer.  Remember to write it back out if
 *		it changes! (ie - a file is changed/deleted/added)
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the directory was read alright...FALSE otherwise.
 *
 * NOTES:	- the label must have been read before this routine is
 *		  called.
 ************************************************************************/
int
hdos_dir_read(struct hdosfs	*hdosfs)
{
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			 lsec = HDOS_DIRLOC(hdosfs);	/* 1st dir block */
    char		 buffer[512];			/* dir block buf */
    int			 count = 0;

    hdir->blocks = count;		/* pretend we don't have any	*/

    while (lsec != 0) {

	if (!hdos_getlsect(hdosfs,lsec,buffer) ||
	    !hdos_getlsect(hdosfs,lsec+1,buffer+256)) {
		return(FALSE);
	}

	hdir->blocks++;
	if (!hdos_dir_blocks_alloc(hdir)) {
	    return(FALSE);
	}

	if (!hdos_dir_block_unpack(&hdir->block[hdir->blocks-1],buffer)) {
	    return(FALSE);
	}

	lsec = hdir->block[hdir->blocks-1].next;
    }

    return(TRUE);
}
int
hdos_dir_write(struct hdosfs	*hdosfs)
{
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			 lsec = HDOS_DIRLOC(hdosfs);	/* 1st dir block */
    char		 buffer[512];			/* dir block buf */
    int			 count = hdir->blocks;

    for (count = 0; count < hdir->blocks; count++) {

	if (!hdos_dir_block_pack(&hdir->block[count],buffer)) {
	    return(FALSE);
	}

	if (!hdos_putlsect(hdosfs,lsec,buffer) ||
	    !hdos_putlsect(hdosfs,lsec+1,buffer+256)) {
		return(FALSE);
	}

	lsec = hdir->block[count].next;
    }
    return(TRUE);
}
    
/************************************************************************
 * NAME:	hdos_name_pack() & hdos_name_unpack()
 *
 * DESCR:	Pack the given name into a directory entry structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- base is assumed to be 8 chars and ext 3 chars
 *		- name is normally in LC and otherwise UC
 *		- name must be \0 terminated
 *		- it appears that in the directory, names are NOT space
 *		  padded, they are NULL padded.  For historical reasons,
 *		  the unpack will deal with space padding.
 ************************************************************************/
static void
hdos_name_pack(char *name, char *base, char *ext)
{
    fs_filename_split(fs_basename(name), base, ext, TRUE);
}

static void
hdos_name_unpack(char *name, char *base, char *ext)
{
    fs_filename_combine(name, base, ext);
}

/************************************************************************
 * NAME:	hdos_dirent()
 *
 * DESCR:	Returns a pointer to the hdos_dirent of the given inode.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
struct hdos_dirent *
hdos_dirent(struct hdosfs *hdosfs, hdos_inode inode)
{
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			block_no = HDOS_INODE_BLOCK(inode);
    int			entry_no = HDOS_INODE_ENTRY(inode);

    return(&hdir->block[block_no].entry[entry_no]);
}

/************************************************************************
 * NAME:	hdos_dir_filedel()
 *
 * DESCR:	Delete the given file from the disk, and return its
 *		groups to the free list (the RGT is updated for the
 *		heck of it, too).
 *
 * ARGS:	the hdosfs_inode
 *
 * RETURNS:	TRUE if it was deleted, FALSE otherwise (file no exist)
 *
 * NOTES:	
 ************************************************************************/
void
hdos_dir_filedel(struct hdosfs *hdosfs, hdos_inode inode)
{
    struct hdos_dirent *entry = hdos_dirent(hdosfs,inode);

    HDOS_MARK_ENTRY_UNUSED(entry);
    hdos_grt_free_chain(hdosfs,entry->firstgroup);
}

/************************************************************************
 * NAME:	hdos_dir_update()
 *
 * DESCR:	Updates a particular directory entry with new information
 *		regarding groups in use, and number of sectors in the
 *		last group.  The modification date is updated too.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
hdos_dir_update_entry(struct hdosfs	*hdosfs, 
		      hdos_inode	inode,
		      int		first,
		      int		last,
		      int		sectors)
{
    struct hdos_dirent	*entry = hdos_dirent(hdosfs,inode);

    entry->firstgroup = first;
    entry->lastgroup = last;
    entry->lastsecidx = sectors;

    hdos_date_now(&entry->mod);
}

/************************************************************************
 * NAME:	hdos_dir_create()
 *
 * DESCR:	Create a new directory entry with the given parameters.
 *
 * ARGS:	
 *
 * RETURNS:	the inode of the new entry, INODE_NULL if none available.
 *
 * NOTES:	
 ************************************************************************/
hdos_dir_create(struct hdosfs *hdosfs,
		char		*name,
		int		 is_system,
		int		 is_locked,
		int		 is_wp,
		int		 is_contig)
{
    struct hdos_dirent	*entry;
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			i, j;
    char		filename[13];

    for (i=0; i < hdir->blocks; i++) {
	for (j=0; j < hdir->block[i].entries; j++) {
	    entry = &hdir->block[i].entry[j];

	    if (HDOS_ENTRY_UNUSED(entry)) {
		goto found;		/* sorry for the goto!	*/
	    }

	    if (HDOS_ENTRY_UNUSED_LAST(entry)) {
		goto found;		/* sorry for the goto!	*/
	    }
	}
    }

 found:
    if ( i == hdir->blocks ) {
	return(HDOS_INODE_NULL);	/* no more free entries	*/
    }


    /* it is likely that the next last doesn't have to be updated	*/
    /* it appears that after a "last" all next entries are marked "last"*/
    /* too.  For now we assume this.					*/

#ifdef NOTDEF
    if (HDOS_ENTRY_UNUSED_LAST(entry)) {	/* need to update "last" */
    	int	next_i, next_j;
    
    	if (i != hdir->blocks - 1 || j != hdir->block[i].entries - 1) {
	    if ( j == hdir->block[i].entries - 1) {
		next_i = i + 1;
		next_j = 0;
	    } else {
		next_i = i;
		next_j = j + 1;
	    }
	    HDOS_MARK_ENTRY_UNUSED_LAST(&hdir->block[next_i].entry[next_j]);
	}
    }
#endif

    /* note that packing the name into the entry unmarks it as "unused"	*/

    hdos_name_pack(name, entry->name, entry->ext);

    entry->is_system = is_system;
    entry->is_locked = is_locked;
    entry->is_wp = is_wp;
    entry->is_contig = is_contig;

    hdos_date_now(&entry->create);

    return(HDOS_INODE(i,j));
}

/************************************************************************
 * NAME:	hdos_dir_find()
 *
 * DESCR:	Finds the given filename in the directory.
 *
 * ARGS:	
 *
 * RETURNS:	The inode of the file if it is found, otherwise the NULL
 *		inode is returned.
 *
 * NOTES:	
 ************************************************************************/
hdos_inode
hdos_dir_find(struct hdosfs *hdosfs, char *filename)
{
    int			i, j;
    char		target[13];
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    struct hdos_dirent	*entry;

    for (i=0; i < hdir->blocks; i++) {
	for (j=0; j < hdir->block[i].entries; j++) {

	    entry = &hdir->block[i].entry[j];

	    if (!HDOS_ENTRY_UNUSED(entry)) {

		hdos_name_unpack(target,entry->name,entry->ext);

		if (strcmp(target,filename) == 0) {
		    return(HDOS_INODE(i,j));
		}
	    }
	}
    }
    return(HDOS_INODE_NULL);
}


/************************************************************************
 * NAME:	hdos_dir_report()
 *
 * DESCR:	Generate a report of the current directory.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
hdos_dir_report(struct hdosfs *hdosfs, int verbosity)
{
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			i, j;
    int			totalgroups = 0;
    int			totalsize = 0;
    char		filename[13];
    unsigned char	tmpRGT[256];
    char		buffer[100];

    if (verbosity >= 3) {
	if (HDOS_NORGT(hdosfs)) {
	    M("No RGT this disk\n");
	}
    }

    printf("Filename     sectors type  S   E   I   Create       Mod   \n");
    printf("------------ ------- ---- --- --- -- ---------- ----------\n");

    for (i=0; i < hdir->blocks; i++) {
	for (j=0; j < hdir->block[i].entries; j++) {
	    struct hdos_dirent *entry = &hdir->block[i].entry[j];

	    if (!HDOS_ENTRY_UNUSED(entry) && !HDOS_ENTRY_UNUSED_LAST(entry)) {
		int	groups;
		int	size;

		hdos_name_unpack(filename,entry->name,entry->ext);

		groups = hdos_dir_file_size(hdosfs,entry);
		size = (groups - 1)*HDOS_GROUPSIZE(hdosfs) + entry->lastsecidx;

		totalgroups += groups;
		totalsize += size;

		printf("%-12.12s ",filename);
		printf("  %3d   ",size);
		printf("%s",entry->is_system?"S":"-");
		printf("%s",entry->is_locked?"L":"-");
		printf("%s",entry->is_wp?"W":"-");
		printf("%s",entry->is_contig?"C":"-");
		printf(" ");
		printf("%3d %3d %2d ", entry->firstgroup, entry->lastgroup,
				  			 entry->lastsecidx);
		hdos_date_format(&entry->create,buffer,0);
		printf("%s ",buffer);
		hdos_date_format(&entry->mod,buffer,0);
		printf("%s\n",buffer);
	    }
	}
    }

    {
	int bytesfree = hdos_grt_free_groups(hdosfs)*HDOS_GROUPSIZE(hdosfs)*HDOS_SECTSIZE(hdosfs);

	printf( "Free space: %d group%s (%s%dk)\n", 
			  hdos_grt_free_groups(hdosfs),
			  (hdos_grt_free_groups(hdosfs)==1)?"":"s",
			  (bytesfree < 1024 && bytesfree!= 0)?".":"",
			  (bytesfree < 1024)?bytesfree/100:bytesfree/1024);
    }
}

/************************************************************************
 * NAME:	hdos_dir_file_size()
 *
 * DESCR:	Returns the size of the given directory entry in groups.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_dir_file_size(struct hdosfs *hdosfs, struct hdos_dirent *entry)
{
    int	g = entry->firstgroup;
    int count = 0;
    do {
	count++;
        g = hdosfs->GRT[g];
    } while (g != 0);

    return(count);
}

/************************************************************************
 * NAME:	hdos_dir_blocks_free()
 *
 * DESCR:	Free the storage allocated to directory entries.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
hdos_dir_blocks_free(struct hdos_dir *hdir) 
{
    free(hdir->block);
    hdos_dir_init(hdir);
}


/************************************************************************
 * NAME:	hdos_dir_init()
 *
 * DESCR:	Initialize the given directory.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
hdos_dir_init(struct hdos_dir *hdir)
{
    hdir->block = NULL;
    hdir->blocks = 0;
    hdir->maxblocks = 0;
}

void
hdos_dir_cleanup(struct hdos_dir *hdir)
{
    hdos_dir_blocks_free(hdir);
}

